/*
 * Decompiled with CFR 0.152.
 */
package org.openslx.filetransfer.util;

import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.zip.CRC32;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.openslx.filetransfer.LocalChunkSource;
import org.openslx.filetransfer.util.ChunkStatus;
import org.openslx.filetransfer.util.FileChunk;
import org.openslx.util.ThriftUtil;

public class ChunkList {
    private static final Logger LOGGER = LogManager.getLogger(ChunkList.class);
    private final List<FileChunk> allChunks;
    private final LinkedList<FileChunk> missingChunks = new LinkedList();
    private final List<FileChunk> pendingChunks = new LinkedList<FileChunk>();
    private final List<FileChunk> completeChunks = new ArrayList<FileChunk>(100);
    private final ByteBuffer statusArray;
    private boolean hasChecksum = false;

    public ChunkList(long fileSize, List<byte[]> sha1Sums) {
        FileChunk.createChunkList(this.missingChunks, fileSize, sha1Sums);
        this.statusArray = ByteBuffer.allocate(this.missingChunks.size());
        this.allChunks = Collections.unmodifiableList(new ArrayList<FileChunk>(this.missingChunks));
    }

    public synchronized int updateSha1Sums(List<byte[]> sha1Sums) {
        int index = 0;
        int firstNew = -1;
        for (byte[] sum : sha1Sums) {
            if (index >= this.allChunks.size()) break;
            if (sum != null) {
                FileChunk chunk = this.allChunks.get(index);
                if (chunk.setSha1Sum(sum)) {
                    if (firstNew == -1) {
                        firstNew = index;
                    }
                    if (chunk.status == ChunkStatus.MISSING && Arrays.equals(FileChunk.NULL_BLOCK_SHA1, sum)) {
                        this.markMissingAsComplete(index);
                    }
                }
                if (!this.hasChecksum) {
                    this.hasChecksum = true;
                }
            }
            ++index;
        }
        return firstNew;
    }

    public synchronized byte[] getDnbd3Crc32List() throws IllegalStateException {
        byte[] buffer = new byte[this.allChunks.size() * 4 + 4];
        long nextChunkOffset = 0L;
        int nextCrcArrayPos = 4;
        for (FileChunk c : this.allChunks) {
            if (c.crc32 == null) {
                throw new IllegalStateException("Called on ChunkList that doesn't have crc32 enabled");
            }
            if (c.range.startOffset != nextChunkOffset) {
                throw new IllegalStateException("Chunk list is not in order or has wrong chunk size");
            }
            nextChunkOffset += 0x1000000L;
            c.getCrc32Le(buffer, nextCrcArrayPos);
            nextCrcArrayPos += 4;
        }
        CRC32 masterCrc = new CRC32();
        masterCrc.update(buffer, 4, buffer.length - 4);
        int value = (int)masterCrc.getValue();
        buffer[3] = (byte)(value >>> 24);
        buffer[2] = (byte)(value >>> 16);
        buffer[1] = (byte)(value >>> 8);
        buffer[0] = (byte)value;
        return buffer;
    }

    public synchronized FileChunk getMissing() throws InterruptedException {
        if (this.missingChunks.isEmpty() && this.pendingChunks.isEmpty()) {
            return null;
        }
        if (this.missingChunks.isEmpty()) {
            this.wait(6000L);
            if (this.missingChunks.isEmpty()) {
                return null;
            }
        }
        FileChunk c = this.missingChunks.removeFirst();
        c.setStatus(ChunkStatus.UPLOADING);
        this.pendingChunks.add(c);
        return c;
    }

    public synchronized boolean hasLocallyMissingChunk() {
        return !this.missingChunks.isEmpty() && this.missingChunks.peekFirst().status == ChunkStatus.MISSING;
    }

    public synchronized FileChunk getCopyCandidate() {
        if (this.missingChunks.isEmpty()) {
            return null;
        }
        FileChunk last = this.missingChunks.removeLast();
        if (last.status != ChunkStatus.QUEUED_FOR_COPY) {
            this.missingChunks.add(last);
            return null;
        }
        last.setStatus(ChunkStatus.COPYING);
        this.pendingChunks.add(last);
        return last;
    }

    public synchronized void markLocalCopyCandidates(List<LocalChunkSource.ChunkSource> sources) {
        for (LocalChunkSource.ChunkSource src : sources) {
            try {
                if (src.sourceCandidates.isEmpty()) continue;
                ArrayList<FileChunk> append = null;
                Iterator it = this.missingChunks.iterator();
                while (it.hasNext()) {
                    FileChunk chunk = (FileChunk)it.next();
                    if (!Arrays.equals(chunk.sha1sum, src.sha1sum) || chunk.status == ChunkStatus.QUEUED_FOR_COPY) continue;
                    if (append == null) {
                        append = new ArrayList<FileChunk>(20);
                    }
                    it.remove();
                    chunk.setStatus(ChunkStatus.QUEUED_FOR_COPY);
                    chunk.setSource(src);
                    append.add(chunk);
                }
                if (append == null) continue;
                this.missingChunks.addAll(append);
            }
            catch (Exception e) {
                LOGGER.warn("chunk clone list is messed up", (Throwable)e);
            }
        }
    }

    public synchronized ByteBuffer getStatusArray() {
        byte[] array = this.statusArray.array();
        for (int i = 0; i < array.length; ++i) {
            FileChunk chunk = this.allChunks.get(i);
            ChunkStatus status = chunk.getStatus();
            array[i] = this.hasChecksum && status == ChunkStatus.COMPLETE && chunk.getSha1Sum() == null ? ChunkStatus.HASHING.val : chunk.getStatus().val;
        }
        return this.statusArray;
    }

    public synchronized List<FileChunk> getCompleted() {
        return new ArrayList<FileChunk>(this.completeChunks);
    }

    public synchronized void resumeFromStatusList(List<Boolean> statusList, long fileLength) {
        if (!this.completeChunks.isEmpty() || !this.pendingChunks.isEmpty()) {
            LOGGER.warn("Inconsistent state: resume called when not all chunks are marked missing");
        }
        int index = 0;
        for (Boolean missing : statusList) {
            FileChunk chunk = this.allChunks.get(index);
            if (fileLength != -1L && fileLength < chunk.range.endOffset) break;
            if (this.missingChunks.remove(chunk) || this.pendingChunks.remove(chunk)) {
                this.completeChunks.add(chunk);
            }
            if (missing.booleanValue()) {
                chunk.setStatus(ChunkStatus.HASHING);
            } else {
                chunk.setStatus(ChunkStatus.COMPLETE);
            }
            ++index;
        }
    }

    public synchronized FileChunk getUnhashedComplete() {
        Iterator<FileChunk> it = this.completeChunks.iterator();
        while (it.hasNext()) {
            FileChunk chunk = it.next();
            if (chunk.sha1sum == null || chunk.status != ChunkStatus.HASHING) continue;
            it.remove();
            this.pendingChunks.add(chunk);
            return chunk;
        }
        return null;
    }

    public synchronized void markCompleted(FileChunk c, boolean hashCheckSuccessful) {
        if (!this.pendingChunks.remove(c)) {
            LOGGER.warn("Inconsistent state: markSuccessful called for Chunk " + c.toString() + ", but chunk is not marked as currently transferring!");
            return;
        }
        c.setStatus(hashCheckSuccessful || c.getSha1Sum() == null ? ChunkStatus.COMPLETE : ChunkStatus.HASHING);
        this.completeChunks.add(c);
        this.notifyAll();
    }

    public synchronized int markFailed(FileChunk c) {
        if (!this.pendingChunks.remove(c)) {
            LOGGER.warn("Inconsistent state: markFailed called for Chunk " + c.toString() + ", but chunk is not marked as currently transferring!");
            return -1;
        }
        c.setStatus(ChunkStatus.MISSING);
        this.missingChunks.addFirst(c);
        this.notifyAll();
        return c.incFailed();
    }

    private synchronized boolean markMissingAsComplete(int index) {
        FileChunk chunk = this.allChunks.get(index);
        if (this.completeChunks.contains(chunk)) {
            return true;
        }
        if (!this.missingChunks.remove(chunk)) {
            LOGGER.warn("Inconsistent state: markMissingAsComplete called for chunk " + chunk.toString() + " (indexed as " + index + ") which is not missing");
            return false;
        }
        chunk.setStatus(ChunkStatus.COMPLETE);
        this.completeChunks.add(chunk);
        this.notifyAll();
        return true;
    }

    public synchronized boolean isComplete() {
        if (!this.missingChunks.isEmpty() || !this.pendingChunks.isEmpty()) {
            return false;
        }
        boolean sawWithHash = false;
        for (FileChunk chunk : this.completeChunks) {
            if (chunk.status == ChunkStatus.HASHING) {
                return false;
            }
            if (chunk.sha1sum != null) {
                sawWithHash = true;
                continue;
            }
            if (chunk.sha1sum != null || !sawWithHash) continue;
            return false;
        }
        return true;
    }

    public synchronized String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append('{');
        for (FileChunk chunk : this.allChunks) {
            sb.append('[');
            sb.append(chunk.getChunkIndex());
            if (chunk.getSha1Sum() != null) {
                sb.append('+');
            }
            switch (chunk.status) {
                case COMPLETE: {
                    sb.append('C');
                    break;
                }
                case COPYING: {
                    sb.append('>');
                    break;
                }
                case HASHING: {
                    sb.append('H');
                    break;
                }
                case MISSING: {
                    sb.append('M');
                    break;
                }
                case QUEUED_FOR_COPY: {
                    sb.append('Q');
                    break;
                }
                case UPLOADING: {
                    sb.append('P');
                    break;
                }
                default: {
                    sb.append('?');
                }
            }
            sb.append('|');
            if (this.missingChunks.contains(chunk)) {
                sb.append('M');
            }
            if (this.pendingChunks.contains(chunk)) {
                sb.append('P');
            }
            if (this.completeChunks.contains(chunk)) {
                sb.append('C');
            }
            sb.append(']');
        }
        sb.append('}');
        return sb.toString();
    }

    public synchronized String getStats() {
        int complete = 0;
        int copying = 0;
        int hashing = 0;
        int missing = 0;
        int qfc = 0;
        int uploading = 0;
        for (FileChunk chunk : this.allChunks) {
            switch (chunk.status) {
                case COMPLETE: {
                    ++complete;
                    break;
                }
                case COPYING: {
                    ++copying;
                    break;
                }
                case HASHING: {
                    ++hashing;
                    break;
                }
                case MISSING: {
                    ++missing;
                    break;
                }
                case QUEUED_FOR_COPY: {
                    ++qfc;
                    break;
                }
                case UPLOADING: {
                    ++uploading;
                }
            }
        }
        return "(" + this.allChunks.size() + ":" + this.completeChunks.size() + "/" + this.pendingChunks.size() + "/" + this.missingChunks.size() + ") (" + complete + "/" + copying + "/" + hashing + "/" + missing + "/" + qfc + "/" + uploading + ")";
    }

    public synchronized boolean isEmpty() {
        return this.allChunks.isEmpty();
    }

    public List<FileChunk> getAll() {
        return this.allChunks;
    }

    public synchronized String getQueueName(FileChunk chunk) {
        if (this.missingChunks.contains(chunk)) {
            return "missing";
        }
        if (this.pendingChunks.contains(chunk)) {
            return "pending";
        }
        if (this.completeChunks.contains(chunk)) {
            return "completed";
        }
        return "NOQUEUE";
    }

    public static boolean hashListsEqualFcBb(List<FileChunk> one, List<ByteBuffer> two) {
        return ChunkList.hashListsEqualFcArray(one, ThriftUtil.unwrapByteBufferList(two));
    }

    public static boolean hashListsEqualFcArray(List<FileChunk> one, List<byte[]> two) {
        if (one.size() != two.size()) {
            return false;
        }
        FileChunk first = one.get(0);
        if (first == null || first.getSha1Sum() == null) {
            return false;
        }
        Iterator<byte[]> it = two.iterator();
        for (FileChunk existingChunk : one) {
            byte[] testChunk = it.next();
            if (Arrays.equals(testChunk, existingChunk.getSha1Sum())) continue;
            return false;
        }
        return true;
    }

    public static boolean hashListsEqualBbBb(List<ByteBuffer> list1, List<ByteBuffer> list2) {
        return ChunkList.hashListsEqualBbArray(list1, ThriftUtil.unwrapByteBufferList(list2));
    }

    public static boolean hashListsEqualBbArray(List<ByteBuffer> bufferList, List<byte[]> arrayList) {
        return ChunkList.hashListsEqualArray(ThriftUtil.unwrapByteBufferList(bufferList), arrayList);
    }

    public static boolean hashListsEqualArray(List<byte[]> list1, List<byte[]> list2) {
        if (list1.size() != list2.size()) {
            return false;
        }
        Iterator<byte[]> it1 = list1.iterator();
        Iterator<byte[]> it2 = list2.iterator();
        while (it1.hasNext() && it2.hasNext()) {
            if (Arrays.equals(it1.next(), it2.next())) continue;
            return false;
        }
        return true;
    }

    public boolean lastChunkIsZero() {
        if (this.allChunks.isEmpty()) {
            return false;
        }
        FileChunk chunk = this.allChunks.get(this.allChunks.size() - 1);
        return chunk.sha1sum != null && Arrays.equals(FileChunk.NULL_BLOCK_SHA1, chunk.sha1sum);
    }

    public void writeCrc32List(String crcfile) throws IllegalStateException, IOException {
        byte[] dnbd3Crc32List = null;
        dnbd3Crc32List = this.getDnbd3Crc32List();
        if (dnbd3Crc32List != null) {
            try (FileOutputStream fos = new FileOutputStream(crcfile);){
                fos.write(dnbd3Crc32List);
            }
        }
    }
}

